<?php

namespace Webkul\UVDesk\AutomationBundle\EventListener;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\EventDispatcher\GenericEvent;
use Webkul\UVDesk\AutomationBundle\Entity\Workflow;
use Webkul\UVDesk\CoreFrameworkBundle\Entity\Ticket;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Webkul\UVDesk\AutomationBundle\Workflow\Event as WorkflowEvent;
use Webkul\UVDesk\AutomationBundle\Workflow\Action as WorkflowAction;
use UVDesk\CommunityPackages\UVDesk\CustomFields\Entity\TicketCustomFieldsValues;


class WorkflowListener
{
    private $container;
    private $entityManager;
    private $registeredWorkflowEvents = [];
    private $registeredWorkflowActions = [];

    public function __construct(ContainerInterface $container, EntityManagerInterface $entityManager)
    {
        $this->container = $container;
        $this->entityManager = $entityManager;
    }

    public function registerWorkflowEvent(WorkflowEvent $serviceTag)
    {
        $this->registeredWorkflowEvents[] = $serviceTag;
    }

    public function registerWorkflowAction(WorkflowAction $serviceTag)
    {
        $this->registeredWorkflowActions[] = $serviceTag;
    }

    public function getRegisteredWorkflowEvent($eventId)
    {
        foreach ($this->registeredWorkflowEvents as $workflowDefinition) {
            if ($workflowDefinition->getId() == $eventId) {
                /*
                    @NOTICE: Events 'uvdesk.agent.forgot_password', 'uvdesk.customer.forgot_password' will be deprecated 
                    onwards uvdesk/automation-bundle:1.0.2 and uvdesk/core-framework:1.0.3 releases and will be 
                    completely removed with the next major release.

                    Both the events have been mapped to return the 'uvdesk.user.forgot_password' id, so we need to 
                    return the correct definition.
                */
                if ('uvdesk.user.forgot_password' == $eventId) {
                    if (
                        $workflowDefinition instanceof \Webkul\UVDesk\CoreFrameworkBundle\Workflow\Events\Agent\ForgotPassword 
                        || $workflowDefinition instanceof \Webkul\UVDesk\CoreFrameworkBundle\Workflow\Events\Customer\ForgotPassword
                    ) {
                        continue;
                    }
                }

                return $workflowDefinition;
            }
        }

        return null;
    }
    
    public function getRegisteredWorkflowEvents()
    {
        return $this->registeredWorkflowEvents;
    }

    public function getRegisteredWorkflowActions()
    {
        return $this->registeredWorkflowActions;
    }

    public function executeWorkflow(GenericEvent $event)
    {
        //$workflowCollection = $this->entityManager->getRepository(Workflow::class)->getEventWorkflows($event->getSubject());
        $raw = $event->getArgument('customfields') ?: [];
        $customFields = [];
        $conn = $this->entityManager->getConnection();

        function flatten($array) {
            $flat = [];
            array_walk_recursive($array, function($a) use (&$flat) {
                $flat[] = $a;
            });
            return $flat;
        }

        foreach ($raw as $fieldId => $jsonIds) {
            if (empty($jsonIds)) continue;

            $ids = is_string($jsonIds) ? json_decode($jsonIds, true) : (is_array($jsonIds) ? $jsonIds : []);
            $ids = array_filter(flatten($ids), static fn($v) => $v !== '' && $v !== null);
            if (empty($ids)) continue;

            $placeholders = implode(',', array_fill(0, count($ids), '?'));
            $sql = "SELECT name FROM uv_pkg_uvdesk_form_component_custom_fields_values WHERE id IN ($placeholders)";
            $names = $conn->fetchFirstColumn($sql, $ids);

            $customFields[$fieldId] = $names;
        }

        $entity = $event->getArgument('entity');
        if (!$entity) return;

        $ticketTypeId = (string) ($entity->getType()?->getId() ?? '');
        if (empty($ticketTypeId)) {
            $cfKeys = array_keys($customFields);
            if (!empty($cfKeys)) {
                $cfId = (int)$cfKeys[0];
                $fallbackId = $this->entityManager->getConnection()->fetchOne(
                    'SELECT ticket_type_id FROM uv_pkg_uvdesk_form_component_custom_fields_types WHERE custom_fields_id = ?',
                    [$cfId]
                );
                $ticketTypeId = (string) $fallbackId;
            }
        }

        $ticketStatusId = (string) ($entity->getStatus()?->getId() ?? '');
        $eventName = $event->getSubject();

        $repo = $this->entityManager->getRepository(Workflow::class);
        if (! $repo instanceof \Webkul\UVDesk\AutomationBundle\Repository\WorkflowRepository) {
            throw new \RuntimeException('Wrong repo: ' . get_class($repo));
        }

        // ✅ First attempt: optimized filtering
        $ids = $repo->findCandidateWorkflowIds($eventName, $ticketTypeId, $ticketStatusId, $customFields);

        // ✅ Fallback to full scan if no workflows matched
        $workflowCollection = [];

            if (!empty($ids)) {
                $workflowCollection = $repo->findBy(['id' => $ids], ['sortOrder' => 'ASC']);
            } else {
                $workflowCollection = $repo->getEventWorkflows($eventName);
            }


        /*
            @NOTICE: Events 'uvdesk.agent.forgot_password', 'uvdesk.customer.forgot_password' will be deprecated 
            onwards uvdesk/automation-bundle:1.0.2 and uvdesk/core-framework:1.0.3 releases and will be 
            completely removed with the next major release.

            From uvdesk/core-framework:1.0.3 onwards, instead of the above mentioned events, the one being 
            triggered will be 'uvdesk.user.forgot_password'. Since there still might be older workflows 
            configured to work on either of the two deprecated events, we will need to make an educated guess 
            which one to use (if any) if there's none found for the actual event.
        */

        if (empty($workflowCollection) && 'uvdesk.user.forgot_password' == $eventName) {
            $user = $event->getArgument('entity');
            if ($user instanceof \Webkul\UVDesk\CoreFrameworkBundle\Entity\User) {
                $agentWFs = $repo->getEventWorkflows('uvdesk.agent.forgot_password');
                $customerWFs = $repo->getEventWorkflows('uvdesk.customer.forgot_password');

                if (!empty($customerWFs) && $user->getCustomerInstance()) {
                    $workflowCollection = $customerWFs;
                } elseif (!empty($agentWFs) && $user->getAgentInstance()) {
                    $workflowCollection = $agentWFs;
                }
            }
        }

        if (!empty($workflowCollection)) {
            foreach ($workflowCollection as $workflow) {
                $totalConditions = 0;
                $totalEvaluatedConditions = 0;

                foreach ($this->evaluateWorkflowConditions($workflow) as $workflowCondition) {
                    $totalEvaluatedConditions++;

                    if (isset($workflowCondition['type']) &&
                        $this->checkCondition($workflowCondition, $entity, $event->getArgument('customfields'), $event)) {
                        $totalConditions++;
                    }

                    if (isset($workflowCondition['or'])) {
                        foreach ($workflowCondition['or'] as $orCondition) {
                            if ($this->checkCondition($orCondition, $entity, $event->getArgument('customfields'), $event)) {
                                $totalConditions++;
                            }
                        }
                    }
                }

                if ($totalEvaluatedConditions === 0 || $totalConditions >= $totalEvaluatedConditions) {
                    $this->applyWorkflowActions($workflow, $event->getArgument('entity'), $event->hasArgument('thread') ? $event->getArgument('thread') : null);
                }
            }
        }
    }


    private function evaluateWorkflowConditions(Workflow $workflow)
    {
        $index = -1;
        $workflowConditions = [];

        if ($workflow->getConditions() == null) {
            return $workflowConditions;
        }

        foreach ($workflow->getConditions() as $condition) {
            if (!empty($condition['operation']) && $condition['operation'] != "&&") {
                if (!isset($finalConditions[$index]['or'])) {
                    $finalConditions[$index]['or'] = [];
                }

                $workflowConditions[$index]['or'][] = $condition;
            } else {
                $index++;
                $workflowConditions[] = $condition;
            }
        }

        return $workflowConditions;
    }

    private function applyWorkflowActions(Workflow $workflow, $entity, $thread = null)
    {
        foreach ($workflow->getActions() as $attributes) {
            if (empty($attributes['type'])) {
                continue;
            }

            foreach ($this->getRegisteredWorkflowActions() as $workflowAction) {
                if ($workflowAction->getId() == $attributes['type']) {
                    $workflowAction->applyAction($this->container, $entity, isset($attributes['value']) ? $attributes['value'] : '', $thread);
                }
            }
        }
    }

    public function checkCondition($condition, $entity, $customField = null, $event = null)
    {
        switch ($condition['type']) {
            case 'from_mail':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    return $this->match($condition['match'], $entity->getCustomer()->getEmail(), $condition['value']);
                }

                break;
            case 'to_mail':
                if (isset($condition['value']) && $entity instanceof Ticket && $entity->getMailboxEmail()) {
                    return $this->match($condition['match'], $entity->getMailboxEmail(), $condition['value']);
                }
                
                break;
            case 'subject':
                if (isset($condition['value']) && ($entity instanceof Ticket || $entity instanceof Task)) {
                    return $this->match($condition['match'], $entity->getSubject(), $condition['value']);
                }

                break;
            case 'description':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $reply = $entity->createdThread->getMessage();
                    $reply = rtrim(strip_tags($reply), "\n" );
                    return $this->match($condition['match'], rtrim($reply), $condition['value']);
                }

                break;
            case 'subject_or_description':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $flag = $this->match($condition['match'], $entity->getSubject(), $condition['value']);
                    $createThread = $this->container->get('ticket.service')->getCreateReply($entity->getId(),false);
                    
                    if (!$flag) {
                        $createThread = $this->container->get('ticket.service')->getCreateReply($entity->getId(),false);
                        $createThread['reply'] = rtrim(strip_tags($createThread['reply']), "\n" );

                        $flag = $this->match($condition['match'],$createThread['reply'],$condition['value']);
                    }

                    return $flag;
                }

                break;
            case 'TicketPriority':
                if (isset($condition['value']) && ($entity instanceof Ticket)) {
                    return $this->match($condition['match'], $entity->getPriority()->getId(), $condition['value']);
                }

                break;
            case 'TicketType':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $typeId = $entity->getType() ? $entity->getType()->getId() : 0;
                    return $this->match($condition['match'], $typeId, $condition['value']);
                }

                break;
            case 'TicketStatus':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    return $this->match($condition['match'], $entity->getStatus()->getId(), $condition['value']);
                }

                break;
            case 'stage':
                if (isset($condition['value']) && $entity instanceof Task) {
                    return $this->match($condition['match'], $entity->getStage()->getId(), $condition['value']);
                }

                break;
            case 'source':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    return $this->match($condition['match'], $entity->getSource(), $condition['value']);
                }

                break;
            case 'created':
                if (isset($condition['value']) && ($entity instanceof Ticket || $entity instanceof Task)) {
                    $date = date_format($entity->getCreatedAt(), "d-m-Y h:ia");
                    return $this->match($condition['match'], $date, $condition['value']);
                }

                break;
            case 'agent':
                if (isset($condition['value']) && $entity instanceof Ticket && $entity->getAgent()) {
                    return $this->match($condition['match'], $entity->getAgent()->getId(), (($condition['value'] == 'actionPerformingAgent') ? ($this->container->get('user.service')->getCurrentUser() ? $this->container->get('user.service')->getCurrentUser()->getId() : 0) : $condition['value']));
                }

                break;
            case 'group':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $groupId = $entity->getSupportGroup() ? $entity->getSupportGroup()->getId() : 0;
                    return $this->match($condition['match'], $groupId, $condition['value']);
                }

                break;
            case 'team':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $subGroupId = $entity->getSupportTeam() ? $entity->getSupportTeam()->getId() : 0;
                    return $this->match($condition['match'], $subGroupId, $condition['value']);
                }

                break;
            case 'customer_name':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    $lastThread = $this->container->get('ticket.service')->getTicketLastThread($entity->getId());
                    return $this->match($condition['match'], $entity->getCustomer()->getFullName(), $condition['value']);
                }
                
                break;
            case 'customer_email':
                if (isset($condition['value']) && $entity instanceof Ticket) {
                    return $this->match($condition['match'], $entity->getCustomer()->getEmail(), $condition['value']);
                }

                break;
            case strpos($condition['value'], 'customFields[') == 0:
                    $value = '';
                
                    // Extract custom field ID from condition type
                    $customFieldId = str_replace('custom_field_', '', $condition['type']);

                    // Get thread and ticket from event
                    //$thread = ($event && $event->hasArgument('thread')) ? $event->getArgument('thread') : null;
                    $ticket = ($event && $event->hasArgument('entity')) ? $event->getArgument('entity') : null;

                    //$threadId = $thread ? $thread->getId() : null;
                    $ticketId = $ticket ? $ticket->getId() : null;

                    // Fetch value using either thread_id or ticket_id
                    if ($customFieldId && ($ticketId)) {
                        $query = "SELECT value FROM uv_pkg_uvdesk_form_component_ticket_custom_fields_values 
                                WHERE custom_field_id = :customFieldId 
                                    AND ticket_id = :ticketId
                                ORDER BY thread_id IS NOT NULL DESC
                                LIMIT 1";

                        $stmt = $this->entityManager->getConnection()->prepare($query);
                        $stmt->bindValue(':customFieldId', $customFieldId);
                        //$stmt->bindValue(':threadId', $threadId);
                        $stmt->bindValue(':ticketId', $ticketId);
                        $stmt->execute();
                        $value = $stmt->fetchColumn();
                    }

                    // Fetch field_type from DB
                    $query = "SELECT field_type FROM uv_pkg_uvdesk_form_component_custom_fields WHERE id = :customFieldId";
                    $stmt = $this->entityManager->getConnection()->prepare($query);
                    $stmt->bindValue(':customFieldId', $customFieldId);
                    $stmt->execute();
                    $fieldType = $stmt->fetchColumn();

                    // If field type is radio, select, or checkbox, get display name
                    if (in_array($fieldType, ['radio', 'select', 'checkbox']) && !empty($value)) {
                        // If it's a checkbox, decode value as array of IDs
                        $valueLabels = [];
                    
                        if ($fieldType == 'checkbox') {
                            $valueIds = json_decode($value, true);
                    
                            if (is_array($valueIds)) {
                                $placeholders = implode(',', array_fill(0, count($valueIds), '?'));
                                $query = "SELECT name FROM uv_pkg_uvdesk_form_component_custom_fields_values WHERE id IN ($placeholders)";
                                $stmt = $this->entityManager->getConnection()->prepare($query);
                                $stmt->execute($valueIds);
                                $names = $stmt->fetchAll(\PDO::FETCH_COLUMN);
                    
                                foreach ($names as $name) {
                                    $valueLabels[] = strtolower(preg_replace('/\s+/', '', $name));
                                }
                            }
                        } else {
                            // Radio/select — get single label
                            $query = "SELECT name FROM uv_pkg_uvdesk_form_component_custom_fields_values WHERE id = :cfValue";
                            $stmt = $this->entityManager->getConnection()->prepare($query);
                            $stmt->bindValue(':cfValue', $value);
                            $stmt->execute();
                            $fieldValueName = $stmt->fetchColumn();
                    
                            if ($fieldValueName) {
                                $valueLabels[] = strtolower(preg_replace('/\s+/', '', $fieldValueName));
                            }
                        }
                    
                        // Normalize expected value
                        $normalizedLabels = array_map(
                            function($lbl) {
                                // remove all whitespace, trim quotes, lowercase
                                return strtolower(trim(preg_replace('/\s+/', '', $lbl), '"'));
                            },
                            $valueLabels
                        );
                        
                        $expectedValue = strtolower(
                            trim(preg_replace('/\s+/', '', $condition['value']), '"')
                        );
                    
                        // Match each label
                        //foreach ($valueLabels as $label) {
                            if (strtolower($condition['match']) === 'is') {
                                if ( in_array($expectedValue, $normalizedLabels, true)) {
                                    return true;
                                }
                            }
                            elseif (strtolower($condition['match']) === 'isnot') {
                                if (! in_array($expectedValue, $normalizedLabels, true)) {
                                    return true;
                                }
                            }
                        //}
                    
                        return false;
                    }

                }
            }

    public function match($condition, $haystack, $needle)
    {
        // Filter tags
        if ('string' == gettype($haystack)) {
            $haystack = strip_tags($haystack);
        }

        switch ($condition) {
            case 'is':
                return is_array($haystack) ? in_array($needle, $haystack) : $haystack == $needle;
            case 'isNot':
                return is_array($haystack) ? !in_array($needle, $haystack) : $haystack != $needle;
            case 'contains':
                return strripos($haystack,$needle) !== false ? true : false;
            case 'notContains':
                return strripos($haystack,$needle) === false ? true : false;
            case 'startWith':
                return $needle === "" || strripos($haystack, $needle, -strlen($haystack)) !== FALSE;
            case 'endWith':
                return $needle === "" || (($temp = strlen($haystack) - strlen($needle)) >= 0 && stripos($haystack, $needle, $temp) !== FALSE);
            case 'before':
                $createdTimeStamp = date('Y-m-d', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d', strtotime($needle . " 23:59:59"));

                return $createdTimeStamp < $conditionTimeStamp ? true : false;
            case 'beforeOn':
                $createdTimeStamp = date('Y-m-d', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d', strtotime($needle . " 23:59:59"));

                return ($createdTimeStamp < $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp) ? true : false;
            case 'after':
                $createdTimeStamp = date('Y-m-d', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d', strtotime($needle . " 23:59:59"));

                return $createdTimeStamp > $conditionTimeStamp ? true : false;
            case 'afterOn':
                $createdTimeStamp = date('Y-m-d', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d', strtotime($needle . " 23:59:59"));
                
                return $createdTimeStamp > $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp ? true : false;
            case 'beforeDateTime':
                $createdTimeStamp = date('Y-m-d h:i:s', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d h:i:s', strtotime($needle));

                return $createdTimeStamp < $conditionTimeStamp ? true : false;
            case 'beforeDateTimeOn':
                $createdTimeStamp = date('Y-m-d h:i:s', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d h:i:s', strtotime($needle));

                return ($createdTimeStamp < $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp) ? true : false;
            case 'afterDateTime':
                $createdTimeStamp = date('Y-m-d h:i:s', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d h:i:s', strtotime($needle));

                return $createdTimeStamp > $conditionTimeStamp ? true : false;
            case 'afterDateTimeOn':
                $createdTimeStamp = date('Y-m-d h:i:s', strtotime($haystack));
                $conditionTimeStamp = date('Y-m-d h:i:s', strtotime($needle));

                return $createdTimeStamp > $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp ? true : false;
            case 'beforeTime':
                $createdTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $haystack));
                $conditionTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $needle));

                return $createdTimeStamp < $conditionTimeStamp ? true : false;
            case 'beforeTimeOn':
                $createdTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $haystack));
                $conditionTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $needle));

                return ($createdTimeStamp < $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp) ? true : false;
            case 'afterTime':
                $createdTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $haystack));
                $conditionTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $needle));

                return $createdTimeStamp > $conditionTimeStamp ? true : false;
            case 'afterTimeOn':
                $createdTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $haystack));
                $conditionTimeStamp = date('Y-m-d H:i A', strtotime('2017-01-01' . $needle));

                return $createdTimeStamp > $conditionTimeStamp || $createdTimeStamp == $conditionTimeStamp ? true : false;
            case 'greaterThan':
                return !is_array($haystack) && $needle > $haystack;
            case 'lessThan':
                return !is_array($haystack) && $needle < $haystack;
            default:
                break;
        }

        return false;
    }
}
