<?php

/// Enable full error reporting
error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * Load environment variables from the UVDesk .env file
 */
function loadUvdeskEnv($filePath)
{
    if (!file_exists($filePath)) {
        die(".env file not found at: $filePath");
    }

    $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        if (strpos($line, '=') !== false) {
            list($key, $value) = explode('=', $line, 2);
            $key = trim($key);
            $value = trim(str_replace('"', '', $value)); // Remove quotes if present
            putenv("$key=$value");
            $_ENV[$key] = $value;
            $_SERVER[$key] = $value;
        }
    }
}

// // Path to UVDesk .env file (Update this path as per your UVDesk installation)
// $uvdeskEnvPath = '/var/www/html/uvdesk/.env'; // For Linux servers
 $uvdeskEnvPath = getcwd() . '/.env'; // For Windows XAMPP

// // Load UVDesk environment variables
loadUvdeskEnv($uvdeskEnvPath);

// Get DB credentials from UVDesk .env file
$databaseUrl = getenv('DATABASE_URL') ?: '';

if (!$databaseUrl) {
    die("DATABASE_URL not found in UVDesk .env file!");
}

// Parse DATABASE_URL (mysql://user:password@host:port/dbname)
$parsedUrl = parse_url($databaseUrl);

$host = $parsedUrl['host'] ?? 'localhost';
$dbname = isset($parsedUrl['path']) ? ltrim($parsedUrl['path'], '/') : die("Database name not found!");
$user = $parsedUrl['user'] ?? 'root';
$pass = $parsedUrl['pass'] ?? '';
$port = $parsedUrl['port'] ?? '3306';


// Helpdesk API credentials
// $helpdesk_url = 'http://localhost/public';
// $api_token = 'Z0JR6UUV7ERB1XRECVDUJWCWYUB0RNTZN067CRECH0XUD5FCMHMT7COKI8VFEEVQ';

// Establish database connection
try {
    $dsn = "mysql:host=$host;port=$port;dbname=$dbname";
    $pdo = new PDO($dsn, $user, $pass);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    echo "Connected successfully to UVDesk database.\n";
} catch (PDOException $e) {
    die("Could not connect to the database: " . $e->getMessage());
}

/**
 * Fetch API Token from `uv_api_access_credentials`
 */
function getLatestApiToken($pdo)
{
    $query = "SELECT token FROM uv_api_access_credentials 
              WHERE is_enabled = 1 AND is_expired = 0 
              ORDER BY created_on DESC LIMIT 1";

    $stmt = $pdo->prepare($query);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);

    return $result ? $result['token'] : null;
}

// Get Site API URL from .env
$site_url = getenv('SITE_URL') ?: die("SITE_URL not found in .env file!");

// Get API Token from Database
$api_token = getLatestApiToken($pdo);

if (!$api_token) {
    die("No valid API token found in `uv_api_access_credentials` table!");
}

echo "Using API Token: $api_token\n";

// Fetch the customer list
$query = "SELECT * FROM uv_user WHERE id IN (SELECT user_id FROM uv_user_instance WHERE supportRole_id = 4)";
$stmt = $pdo->query($query);
$customers = $stmt->fetchAll(PDO::FETCH_ASSOC);

if (!$customers) {
    die("No customers found.");
}

// Fetch the Ticket Type ID for 'GR SHARING KNOWLEDGE WITH CLIENT'
// $typeQuery = "SELECT * FROM uv_ticket_type WHERE code = 'GR SHARING KNOWLEDGE WITH CLIENT'";
// $typeStmt = $pdo->query($typeQuery);
// $typeResult = $typeStmt->fetch(PDO::FETCH_ASSOC);
// $ticketTypeId = $typeResult ? $typeResult['id'] : die("Ticket type 'GR SHARING KNOWLEDGE WITH CLIENT' not found!");

//echo "Ticket Type ID: $ticketTypeId\n";

// Loop through customers and create tickets
foreach ($customers as $customer) {
    $previousMonthYear = date('F-Y', strtotime('first day of last month')); // Get last month

    $ticketData = [
        'actAsType' => 'customer',
        'name' => $customer['first_name'] . ' ' . $customer['last_name'],
        'from' => $customer['email'],
        'subject' => "Monthly GST-R Ticket for $previousMonthYear", // Updated Subject
        'message' => "This is your automatically generated Monthly GST-Registration ticket for $previousMonthYear.",
        'type' => 'GST-R1-001 Data Collection', // Using ID instead of name
        'status' => 1,  // Open status
    ];

    // Send request to UVDesk API
    $ch = curl_init($site_url . '/api/v1/ticket');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        "Authorization: Bearer $api_token",
        "Content-Type: application/json",
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($ticketData));

    // 🔴 Disable SSL verification (temporary fix)
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    echo "API Response: " . ($response ? $response : "No Response") . "\n";
    echo "HTTP Code: $httpCode\n";
    echo "CURL Error: $curlError\n";

    if ($httpCode == 200) {
        echo "Ticket created successfully for " . $customer['email'] . "\n";
    } else {
        echo "Failed to create ticket for " . $customer['email'] . "\n";
        echo "HTTP Code: $httpCode\n";
        echo "CURL Error: $curlError\n";
        echo "Response: " . ($response ? $response : "No Response") . "\n";
    }

}

?>
